#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import logging
import json
from distutils.version import LooseVersion
from dataclasses import dataclass
from . import patcher, helpers
from typing import Optional, Dict, Any
import webbrowser
import platform

# As defined in platform.system()
PLATFORM_WINDOWS = "Windows"
PLATFORM_MAC = "Darwin"
PLATFORM_LINUX = "Linux"


@dataclass
class VersionInfo:
    version: LooseVersion
    last_compatible_version: LooseVersion
    files: Dict[str, str]

    def get_file(self) -> Optional[str]:
        return self.files.get(platform.system(), None)

    @staticmethod
    def parse(json_str: str) -> Optional["VersionInfo"]:
        try:
            version_json = json.loads(json_str)
            logging.debug(version_json)

            return VersionInfo(version_json["version"],
                               version_json["last_compatible_version"],
                               version_json["files"])
        except Exception as e:
            logging.debug(e, exc_info=True)
            return None

    def to_json(self) -> Dict[str, Any]:
        return {
            "version": str(self.version),
            "last_compatible_version": str(self.last_compatible_version),
            "files": self.files,
        }


def check_version(rel_url: str, current_version: LooseVersion, builddir: str) -> bool:
    """Check for new patcher versions. Returns true when the program should continue or false otherwise."""
    logging.info("Checking for patcher updates...")

    versiondata = patcher.get_from_server_str(rel_url)
    if not versiondata:
        logging.warning("No version info found")
        return True

    info = VersionInfo.parse(versiondata)
    if not info:
        logging.error("Incomplete version info")
        return True

    if current_version < info.version:
        is_required = current_version < info.last_compatible_version
        helpers.print_separator()
        logging.info("A new patcher version is available: v%s", info.version)
        if is_required:
            logging.info("This update is required in order to proceed.")
        helpers.print_separator()
        if helpers.confirm("Open download link"):
            webbrowser.open(patcher.get_server_url(info.get_file()))

            helpers.print_separator()
            print("Install instructions:")
            print("1. Download the new patcher zip")
            print("2. Extract the zip")
            print(f"3. Move the \"{builddir}\" folder to the new patcher folder")
            print("4. Remove the old patcher")
            helpers.print_separator()
            return False

        if is_required:
            return False
    else:
        logging.info("Patcher is up-to-date")

    return True
