#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import psutil
import os
from typing import List, IO, Optional


def path_sep(host=False):
    return os.sep if host else "/"


def net_path(path: str):
    return path_normalize(path, host=False)


def host_path(path: str):
    return path_normalize(path, host=True)


def path_normalize(path: str, host):
    """Ensures the path has the correct separator."""
    if host and os.sep == "\\":
        path = path.replace("/", "\\")
    else:
        path = path.replace("\\", "/")
    return path.rstrip(path_sep(host))


def path_join(*paths, host=False):
    return path_sep(host).join([ path_normalize(i, host) for i in paths ])


def custom_readlines(handle: IO, chunk_size: int = 64, encoding="utf-8"):
    """Yields lines from IO handle and treats carriage returns as newlines.

    Blocks until EOF.
    """
    buf = ""
    while not handle.closed:
        data = handle.read(chunk_size)
        if not data:
            break

        buf += data.decode(encoding).replace("\r", "\n")
        lines: List[str] = buf.split("\n")
        buf = lines.pop()

        for i in lines:
            yield i

    if buf:
        yield buf


def find_process(exepath) -> Optional[psutil.Process]:
    for i in psutil.process_iter():
        try:
            procexe = os.path.realpath(i.exe())
            if procexe in (exepath, exepath + ".exe"):
                return i
        except psutil.Error:
            pass
    return None


def urljoin(base, *p):
    return net_path(os.path.join(base, *p))


def get_terminal_width() -> int:
    # One less than terminal width to avoid newline on windows
    return os.get_terminal_size()[0] - 1


def confirm(question: str) -> bool:
    while True:
        try:
            ans = input(f"{question}? (y/N) ").lower()
            if ans == "n":
                return False
            if ans == "y":
                return True
        except Exception:  # Don't catch KeyboardInterrupt and SystemExit
            pass


def print_separator() -> None:
    print(get_terminal_width() * "─")
