#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os
import json
import logging
from typing import Dict, List
from . import helpers


class RepoConfig:  # pylint: disable=too-many-instance-attributes
    def __init__(self):
        self._name: str = "build"
        self.url: str = ""
        self._sftp_dir: str = ""
        self.sftp_user: str = ""
        self.sftp_port: int = 22
        self.process_name: str = ""

    @property
    def name(self) -> str:
        return self._name

    @name.setter
    def name(self, value: str):
        # Ensure it's really just a name and not a path.
        self._name = value.replace("\\", "").replace("/", "")

    @property
    def sftp_dir(self):
        return self._sftp_dir

    @sftp_dir.setter
    def sftp_dir(self, value):
        self._sftp_dir = helpers.net_path(value)

    def load(self, json_node: Dict) -> bool:
        self.name         = json_node.get("build", self.name)
        self.sftp_dir     = json_node.get("ftpdir", self.sftp_dir)
        self.sftp_user    = json_node.get("ftpuser", self.sftp_user)
        self.sftp_port    = json_node.get("ftpport", self.sftp_port)
        self.url          = json_node.get("url", self.url)
        self.process_name = json_node.get("wait_for_process", self.process_name)

        if not self.url:
            logging.warning("No URL specified")

        return self.is_valid()

    def is_valid(self) -> bool:
        if not self.name or not self.url:
            return False
        return True


class PatcherConfig:
    DEFAULT_ARIA_DIR = "aria2"

    def __init__(self):
        self.repos: List[RepoConfig] = []
        self.aria_cmd_flags = [
            "--no-conf=true",
            "--allow-overwrite=true",
            "--auto-file-renaming=false",
            "--optimize-concurrent-downloads=true",
            "--remote-time=true",
            "--console-log-level=warn",
            "--download-result=hide",
            "--summary-interval=0",

            # Always start from scratch, to prevent files being continued with a newer
            # upstream version.
            "--continue=false",
            "--remove-control-file=true",
            "--always-resume=false",
            "--auto-save-interval=0",
        ]

    def load(self, fname: str) -> bool:
        if not os.path.isfile(fname):
            logging.error("Config file does not exist: %s", fname)
            return False

        with open(fname) as f:
            config = json.load(f)

        # Repositories
        repos = config.get("repos", None)
        if not repos:
            logging.error("No repos specified")
            return False

        for i in repos:
            repocfg = RepoConfig()
            if repocfg.load(i):
                self.repos.append(repocfg)
            else:
                logging.error("Invalid repo config: %s", repocfg.name)

        # Aria searchpath
        aria_dir = helpers.host_path(config.get("aria", self.DEFAULT_ARIA_DIR))
        if aria_dir:
            os.environ["PATH"] += os.pathsep + os.path.abspath(aria_dir)

        # Download limit
        limit = config.get("limit", 0)
        if limit != 0:
            self.aria_cmd_flags.append("--max-overall-download-limit=" + str(limit))

        return True
