#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import shutil
from subprocess import Popen, PIPE
import sys
import logging
from typing import List
import re
from . import helpers
from .TransferProgress import TransferProgress

ARIA_BINARY = "aria2c"

REGEX_PROGRESS = re.compile(r"\[DL:(.+?)\](\[#.+\])*")
REGEX_PROGRESS_PARTS = re.compile(r"\[#.+?\s(.+?)/.+?(?:\(.+?\))?\]")
REGEX_PROGRESS_SINGLE = re.compile(r"\[#.+?\s(.+?)/.+?\sDL:(.+?B).*\]")
REGEX_FINISHED = re.compile(r".*\[NOTICE\] Download complete: (.*)$")

ERROR_CODES = {
    0: "All downloads were successful.",
    1: "An unknown error occurred.",
    2: "Time out occurred.",
    3: "A resource was not found.",
    4: "Aria2 saw the specified number of \"resource not found\" error. See --max-file-not-found option.",
    5: "A download aborted because download speed was too slow. See --lowest-speed-limit option.",
    6: "Network problem occurred.",
    7: "There were unfinished downloads. This error is only reported if all finished downloads were successful and there were unfinished downloads in a queue when aria2 exited by pressing Ctrl-C by an user or sending TERM or INT signal.",
    8: "Remote server did not support resume when resume was required to complete download.",
    9: "There was not enough disk space available.",
    10: "Piece length was different from one in .aria2 control file. See --allow-piece-length-change option.",
    11: "Aria2 was downloading same file at that moment.",
    12: "Aria2 was downloading same info hash torrent at that moment.",
    13: "File already existed. See --allow-overwrite option.",
    14: "Renaming file failed. See --auto-file-renaming option.",
    15: "Aria2 could not open existing file.",
    16: """Aria2 could not create new file or truncate existing file.
This error is often caused by missing write access.
Try to disable your Anti-Virus software and try again.""",
    17: "File I/O error occurred.",
    18: "Aria2 could not create directory.",
    19: "Name resolution failed.",
    20: "Aria2 could not parse Metalink document.",
    21: "FTP command failed.",
    22: "HTTP response header was bad or unexpected.",
    23: "Too many redirects occurred.",
    24: "HTTP authorization failed.",
    25: "Aria2 could not parse bencoded file (usually \".torrent\" file).",
    26: "\".torrent\" file was corrupted or missing information that aria2 needed.",
    27: "Magnet URI was bad.",
    28: "Bad/unrecognized option was given or unexpected option argument was given.",
    29: "The remote server was unable to handle the request due to a temporary overloading or maintenance.",
    30: "Aria2 could not parse JSON-RPC request.",
    31: "Reserved. Not used.",
    32: "Checksum validation failed.",
}


class AriaError(Exception):
    def __init__(self, code: int):
        self.code: int = code
        errstr = ERROR_CODES.get(self.code, "")
        super().__init__(f"Aria exited with error code {self.code}: {errstr}")


class AriaDownloader:
    def __init__(self):
        self._urls: List[str] = []

    def add_url(self, url: str, outdir: str = ""):
        assert url, "Empty URL specified"
        self._urls.append(url)

        if outdir:
            self._urls.append("\tdir={}".format(outdir))

    def clear(self):
        self._urls.clear()

    def start(self, flags: List[str] = None, progress: TransferProgress = None) -> None:
        """Starts download and returns True on success, otherwise False. Exceptions are reraised."""
        flags = flags or []

        if not shutil.which(ARIA_BINARY):
            raise FileNotFoundError("Can't find aria2 binary")

        # Setup unbuffered stdout, because on mac and linux, the buffer does
        # not flush on \r, which is needed to parse progress and download speed.
        if sys.platform in ("linux", "linux2"):
            args = [ "stdbuf", "-o0" ]
        elif sys.platform == "darwin":
            args = [ "unbuffer" ]
        else:
            args = []

        args.append(ARIA_BINARY)
        args.extend(flags)
        args.append("--input-file=-")

        if progress:
            args.extend([
                "--console-log-level=notice",
                "--download-result=hide",
                "--truncate-console-readout=false",
                "--summary-interval=0",
                "--show-console-readout=true",
                "--enable-color=false",
                "--human-readable=false",
            ])

        intext = "\n".join(self._urls).encode("utf-8")
        logging.debug("aria input:\n%s", intext)

        logging.info("Downloading files...")
        progress.print_progress_bar()

        with Popen(args, stdin=PIPE, stdout=(PIPE if progress else None)) as p:
            try:
                if not progress:
                    p.communicate(intext)
                else:
                    p.stdin.write(intext)
                    p.stdin.close()

                    for i in helpers.custom_readlines(p.stdout):
                        if i:
                            self._parse_aria(progress, i)
                        progress.update_speed(speed=False)
                        progress.print_progress_bar()
            except:  # noqa
                # Make sure to terminate process gracefully
                if p.poll() is None:
                    p.terminate()
                raise

        if p.returncode != 0:
            raise AriaError(p.returncode)

        logging.debug("Aria exited successfully")

    @staticmethod
    def _parse_aria(progress: TransferProgress, line: str):
        m = REGEX_FINISHED.match(line)
        if m:
            progress.mark_finished(m.group(1), True)
            # logging.info("finished: %s", m.group(1))
            return True

        m = REGEX_PROGRESS.match(line)
        if m:
            parts = REGEX_PROGRESS_PARTS.findall(m.group(2))
            # logging.info("read: %s", line)
            # logging.info("speed: %s, parts: %s", speed, parts)
            progress.mark_in_progress([ int(i[:-1]) for i in parts ])
            progress.set_speed(int(m.group(1)[:-1]))
            return True

        m = REGEX_PROGRESS_SINGLE.match(line)
        if m:
            # logging.info("speed: %s, parts: %s", speed, part)
            progress.mark_in_progress([ int(m.group(1)[:-1]) ])
            progress.set_speed(int(m.group(2)[:-1]))
            return True

        return False
