#!/usr/bin/env python3
# -*- coding: utf-8 -*-

from collections import deque
import time
import datetime
import os
import logging
from typing import List, Optional
from statistics import median
from .FileInfo import FileInfo
from . import helpers


class SpeedTimer:
    def __init__(self, progress: "Progress", interval: int = 2):
        self._interval: int = interval
        self._lasttime = time.time()
        self._lastsize = 0
        self._progress = progress
        self._avg: deque = deque([], maxlen=20)
        self._avg_speed: float = 0
        self._lastetatime = self._lasttime

    def get_speed(self) -> float:
        if len(self._avg) > 0:
            return self._avg[-1]
        return 0

    def get_avg_speed(self) -> float:
        return self._avg_speed

    def set_speed(self, speed: float):
        self._avg.append(speed)

    def autoupdate(self, speed=True, eta=True):
        t = time.time()

        if speed:
            delta = t - self._lasttime
            if delta >= self._interval:
                newsize = self._progress.get_progress_bytes()
                self.set_speed((newsize - self._lastsize) / delta)
                self._lasttime = t
                self._lastsize = newsize

        if eta:
            delta = t - self._lastetatime
            if delta >= 10:
                self._lastetatime = t
                self._avg_speed = median(self._avg)
                lastspeed = self.get_speed()
                self._avg.clear()
                self.set_speed(lastspeed)

    def get_eta(self) -> float:
        speed = self.get_avg_speed()
        if not speed:
            return 0
        return (self._progress.get_size() - self._progress.get_progress_bytes()) / speed


class Progress:
    def __init__(self, size: int):
        self._size = size
        self._progress_size: int = 0
        self._speed_timer = SpeedTimer(self)

        self.no_speed_info: bool = False  # Don't print speed in print_progress_bar()

    def get_size(self) -> int:
        return self._size

    def set_speed(self, speed: float):
        self._speed_timer.set_speed(speed)

    def update_speed(self, speed=True, eta=True):
        self._speed_timer.autoupdate(speed, eta)

    def get_speed(self) -> float:
        return self._speed_timer.get_speed()

    def reset(self):
        self._progress_size = 0
        self.set_speed(0)

    def add_progress(self, value: int):
        self._progress_size += value
        self.update_speed()

    def get_progress(self) -> float:
        """Return percentage progress in range 0-1."""
        return self.get_progress_bytes() / max(self._size, 1)

    def get_progress_bytes(self) -> int:
        return self._progress_size

    def finish(self, aborted=False):
        if not aborted:
            self.reset()
            self._progress_size = self._size
        self.print_progress_bar(end="\n")

    def print_progress_bar(self, end: Optional[str] = "\r"):
        stats = "{}{:.2f}%".format(
            "" if self.no_speed_info else "{:.2f} MiB/s - ".format(self.get_speed() / 1024 ** 2),
            self.get_progress() * 100)

        # if self.no_speed_info:
        #     eta = ""
        # else:
        eta = "ETA: {}".format(
            str(datetime.timedelta(seconds=int(self._speed_timer.get_eta()))))

        # one less than terminal width to avoid newline on windows
        width = helpers.get_terminal_width() - len(stats) - len(eta) - 6
        filled = round(self.get_progress() * width) * "█"
        nonfilled = " " * (width - len(filled))

        print(" {} [{}{}] {} ".format(stats, filled, nonfilled, eta), end=end)

    def __enter__(self):
        # It makes sense to have this here, but it could cause printout text
        # to overlap the progress bar before the progress actually starts.
        # self.print_progress_bar()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        self.finish(aborted=exc_type is not None)


class TransferProgress(Progress):
    def __init__(self, filelist: List[FileInfo]):
        self._filelist = filelist
        self._filedict = { i.filename: i for i in filelist }
        self._progress_varying_size: int = 0
        self.print_finished_files: bool = False

        super().__init__(sum(i.size for i in filelist))

    def reset(self):
        super().reset()
        self._progress_varying_size = 0

    def mark_finished(self, filename: str, subtract_in_progress: bool):
        filename = helpers.host_path(filename)
        file: FileInfo = self._filedict.get(filename, None)
        if not file:
            logging.error("File not found in file list: %s", filename)
            return
        self.add_progress(file.size)

        if self.print_finished_files:
            # Make sure to override the progress bar spanning the terminal width
            s = "Completed file {}".format(file.filename)
            tw = os.get_terminal_size()[0] - 1
            padlen = tw - len(s)
            if padlen > 0:
                s += " " * padlen
            logging.info(s)
        else:
            logging.debug("Completed file %s", file.filename)

        if subtract_in_progress:
            self.mark_in_progress([ self._progress_varying_size - file.size ])

    def mark_in_progress(self, progress_sizes: List[int]):
        self._progress_varying_size = sum(progress_sizes)

    def get_progress_bytes(self) -> int:
        return super().get_progress_bytes() + self._progress_varying_size


class PatcherProgress(TransferProgress):
    """Strips the output dir part from paths."""
    def __init__(self, filelist: List[FileInfo], outdir: str):
        super().__init__(filelist)
        self._outdir = helpers.host_path(outdir).rstrip(os.sep) + os.sep

    def mark_finished(self, filename: str, subtract_in_progress: bool):
        # Strip output dir from path
        filename = helpers.host_path(filename)
        if filename.startswith(self._outdir):
            filename = filename[len(self._outdir):]
        super().mark_finished(filename, subtract_in_progress)
