#!/usr/bin/env python3
# -*- coding: utf-8 -*-

from distutils.version import LooseVersion
from typing import List, Optional
import os
import logging
import re
import sys
from patcherlib import patcher, helpers
from patcherlib.patchdiff import Database, FileInfo, DiffList
import version
from datetime import date

PROGRAM_NAME = "OjsUndPatcher"
MAX_PATCHLOGS = 5


class Patchlog:
    PATCHLOG_PATTERN = re.compile(r'(?:(\d{4}-\d{2}-\d{2}).*)?patchlog v(.*)\.txt', re.IGNORECASE)

    def __init__(self, fileinfo_: FileInfo, version_string_: str, isodate: str):
        self.fileinfo = fileinfo_
        self.version_string = version_string_
        self.patch_date = None

        if isodate:
            try:
                self.patch_date = date.fromisoformat(isodate)
            except (ValueError, TypeError) as e:
                logging.error("Failed to parse date: %s", e)

    @staticmethod
    def from_file(file: FileInfo) -> Optional["Patchlog"]:
        name = os.path.split(file.filename)[1]
        if m := Patchlog.PATCHLOG_PATTERN.match(name):
            return Patchlog(file, m.group(2), m.group(1))
        return None

    def update_file(self, db: Database, difflist: DiffList):
        if self.patch_date:
            new_fname = f"{self.patch_date} - patchlog v{self.version_string}.txt"
        else:
            logging.warning("Missing patch date for patch %s", self.version_string)
            new_fname = f"patchlog v{self.version_string}.txt"

        full_path = os.path.join(db.get_path(), self.fileinfo.filename)
        new_full_path = os.path.join(os.path.split(full_path)[0], new_fname)

        if full_path == new_full_path:
            return

        try:
            os.rename(full_path, new_full_path)
            logging.info("Renamed '%s' to '%s'", full_path, new_full_path)
        except Exception as e:
            logging.error("Failed to rename patchlog from '%s' to '%s': %s",
                          full_path, new_full_path, str(e))
            raise

        db.remove_file(self.fileinfo)
        newinfo = db.add_file(new_full_path, is_relative=False)
        difflist.modified.pop(difflist.modified.index(self.fileinfo))
        difflist.modified.append(newinfo)
        db.save(patcher.INDEX_FILE)
        self.fileinfo = newinfo


class CustomRunner(patcher.Runner):
    def _find_new_patchlogs(self) -> List[Patchlog]:
        logging.debug("Retrieving new patchlogs...")

        patchlogs: List[Patchlog] = []

        for i in self._diff.modified:
            if not i.is_directory():
                if patch := Patchlog.from_file(i):
                    patchlogs.append(patch)

        patchlogs.sort(key=lambda x: LooseVersion(x.version_string))
        return patchlogs

    def _print_patchlogs(self):
        patchlogs = self._find_new_patchlogs()
        termwidth = helpers.get_terminal_width()
        first = True

        for patch in patchlogs[-MAX_PATCHLOGS:]:
            data = patcher.get_from_server_str(
                helpers.urljoin(patcher.BUILD_DIR, patch.fileinfo.filename))

            if data is None:
                logging.error("Failed to retrieve patchlog %s.", patch.fileinfo.filename)
                continue

            if first:
                first = False
            else:
                print()

            # We don't really want this to appear in the logs, it has nothing
            # to do with the patcher itself. However, let's log the patch
            # version, in case ordering is broken or something.
            helpers.print_separator()
            logging.info("Patchlog v{} ({})".format(
                patch.version_string,
                patch.patch_date.strftime("%A, %d.%m.%Y") if patch.patch_date else "Unknown Date"
            ).center(termwidth, " "))
            helpers.print_separator()
            print(data.strip())

        if len(patchlogs) > MAX_PATCHLOGS:
            print()
            helpers.print_separator()
            logging.info("%s earlier patchlogs were omitted.", len(patchlogs) - MAX_PATCHLOGS)

        helpers.print_separator()

    def _on_before_upload(self):
        patchlogs: List[Patchlog] = self._find_new_patchlogs()

        for patch in patchlogs:
            if not patch.patch_date:
                patch.patch_date = date.today()
                patch.update_file(self._local, self._diff)

        print()
        super()._on_before_upload()

    def _on_before_download(self):
        self._print_patchlogs()
        print()
        super()._on_before_download()


if __name__ == "__main__":
    sys.exit(CustomRunner(PROGRAM_NAME, version.VERSION).run())
